function torques_forces = force_calculator(obj)
%% 
% Calculates the net force and torque for a given [x y z theta phi] position of the nucleus
% MT force : F = L^Beta1 + pullfactor*L^Beta2 + pullfactor_exp*2^(L/Unit_branch) when touching polarity
% In paper : F = A L^3 + B [.] L^2 + C [.] 2^(L/a)
% Input:    instance of the class "sim_program".
% output:   [torque_theta, torque_phi, force_x, force_y, force_z]
%
% Original Version: Anaelle Pierre, 02/Oct/2015
% Final Release:    Dmitry Ershov,  06/Dec/2017
% Nicolas Minc Team at IJM, Paris


%%
%tic

Shape_mat           = obj.current_cell.shape_mat;
Pol_mat             = obj.current_cell.polarity_mat;
Stab_mat            = obj.current_cell.stab_mat;

% spindle position and orientation
x                   = obj.current_cell.x;
y                   = obj.current_cell.y;
z                   = obj.current_cell.z;
theta_rad           = obj.current_cell.theta * pi/180;   % spindle orientation in radians
phi_rad             = obj.current_cell.phi * pi/180;

% pulling forces
pullfactor          = obj.current_cell.pullfactor;
pullfactor_exp      = obj.current_cell.pullfactor_exp;
Beta1               = obj.current_cell.Beta1; 
Beta2               = obj.current_cell.Beta2;

% lengths:
Nucsize             = obj.current_cell.nucleus_d;           % nucleus size
shape_span_avg      = obj.current_cell.shape_span_avg;      % average length of the cell shape
max_mt_length       = obj.current_cell.shape_span_max;      % maximal allowed MT length = maximal span of the cell shape

% other
anglelimit_frac_of_180  = obj.current_cell.anglelimit_frac_of_180;          % aster angular extension, [fraction of 180] = [degrees/180].
Unit_branch_2           = obj.current_cell.Unit_branch_2;
anglelimit_deg          = obj.current_cell.anglelimit_deg;          % aster angular extension, [fraction of 180] = [degrees/180].



% do we have polarity domain?
if pullfactor > 0 || pullfactor_exp > 0 
    calculating_polarity        = 1;
else
    calculating_polarity        = 0;
end


% centrosome 1
centro1_x              = x + Nucsize/2 * sin(theta_rad) * cos(phi_rad);
centro1_y              = y + Nucsize/2 * sin(theta_rad) * sin(phi_rad);
centro1_z              = z + Nucsize/2 * cos(theta_rad);

% centrosome 2
centro2_x              = x - Nucsize/2 * sin(theta_rad) * cos(phi_rad);
centro2_y              = y - Nucsize/2 * sin(theta_rad) * sin(phi_rad);
centro2_z              = z - Nucsize/2 * cos(theta_rad);



%% Generate a set of polar coordinates
% for ALL MT that will be thrown from the 2 centrosomes.
% polar angle will be from 0 to 180 (alpha, analog of theta)
% azimuthal will be from 0 to 360   (psi, analog of phi)

alpha_step  = obj.angle_btw_mt_deg;            % angle in degrees: a step from one MT to another.
psi_step    = obj.angle_btw_mt_deg;
% alpha_step  = 10;            % test
% psi_step    = 10;

N_psi       = round(360/psi_step);              % number of azimuthal angles
N_alpha     = round(anglelimit_deg/alpha_step);            % number of polar angles

alpha_set_0   = linspace(alpha_step, anglelimit_deg - alpha_step, N_alpha-1);  % middle part of the set
psi_set_0     = linspace(psi_step, 360-psi_step, N_psi-1);        % middle part of the set

alpha_set_deg   = [0, alpha_set_0, anglelimit_deg];               % full set, includeing 0 and 180.
psi_set_deg     = [0, psi_set_0];                 % full set, includeing 0 and 360.

alpha_set_rad   = pi / 180 * alpha_set_deg;
psi_set_rad     = pi / 180 * psi_set_deg;

N_psi       = length(psi_set_deg);                   % number of elements now    
N_alpha     = length(alpha_set_deg);                 % number of elements now 


% make 2D matrices for these angles.
[alphamat_rad, psimat_rad]      = meshgrid(alpha_set_rad, psi_set_rad);         % angles for MT (1:180), (1:360)

% pre-calculate some parameters linked to these angles (this makes
% the program faster).
cos_alphamat            = cos(alphamat_rad);
sin_alphamat            = sin(alphamat_rad);
cos_psimat              = cos(psimat_rad);
sin_psimat              = sin(psimat_rad);





                                           
%% Trace maximal length MTs for each angle (to set directions).
% MTs for the centrosome 1 
% MTs directions for centrosome 1  and 2
theta_rad_1         = theta_rad;
phi_rad_1           = phi_rad;
centro1_mt_ends_x   = centro1_x + max_mt_length * (cos_alphamat .* sin(theta_rad_1) .* cos(phi_rad_1) + ...
                                                   sin_alphamat .* cos_psimat .*cos(phi_rad_1).*cos(theta_rad_1) - ...
                                                   sin_alphamat .* sin_psimat .* sin(phi_rad_1));             
centro1_mt_ends_y   = centro1_y + max_mt_length * (cos_alphamat .* sin(theta_rad_1) .* sin(phi_rad_1) + ...
                                                   sin_alphamat .* cos_psimat .* sin(phi_rad_1) .* cos(theta_rad_1) + ...
                                                   sin_alphamat .* sin_psimat .* cos(phi_rad_1));
centro1_mt_ends_z   = centro1_z + max_mt_length * (cos_alphamat .* cos(theta_rad_1) - ...
                                                   sin_alphamat .* cos_psimat .* sin(theta_rad_1));
                                                  
% MTs for the centrosome 2     
theta_rad_2         = pi - theta_rad;
phi_rad_2           = phi_rad + pi;
centro2_mt_ends_x   = centro2_x + max_mt_length * (cos_alphamat .* sin(theta_rad_2) .* cos(phi_rad_2) + ...
                                                   sin_alphamat .* cos_psimat .* cos(phi_rad_2).*cos(theta_rad_2) - ...
                                                   sin_alphamat .* sin_psimat .* sin(phi_rad_2));
centro2_mt_ends_y   = centro2_y + max_mt_length * (cos_alphamat .* sin(theta_rad_2) .* sin(phi_rad_2) + ...
                                                   sin_alphamat .* cos_psimat .* sin(phi_rad_2) .* cos(theta_rad_2) + ...
                                                   sin_alphamat .* sin_psimat .* cos(phi_rad_2));
centro2_mt_ends_z   = centro2_z + max_mt_length * (cos_alphamat .* cos(theta_rad_2) - ...
                                                   sin_alphamat .* cos_psimat .* sin(theta_rad_2));

                             
% direction of the MT vectors 
% IN CARTESIAN system, RELATIVE TO CENTROSOME (not (0,0,0)!!!)
mt_dir_01_x   = (centro1_mt_ends_x - centro1_x) / max_mt_length;
mt_dir_01_y   = (centro1_mt_ends_y - centro1_y) / max_mt_length;
mt_dir_01_z   = (centro1_mt_ends_z - centro1_z) / max_mt_length;
                                                  
% MTs for the centrosome 2                                                   
mt_dir_02_x   = (centro2_mt_ends_x - centro2_x) / max_mt_length;
mt_dir_02_y   = (centro2_mt_ends_y - centro2_y) / max_mt_length;
mt_dir_02_z   = (centro2_mt_ends_z - centro2_z) / max_mt_length;


% DE DEBUG check what is going on so far:

% max_mt_length_test = 400;
% centro1_mt_ends_x   = centro1_x + max_mt_length_test * mt_dir_01_x;
% centro1_mt_ends_y   = centro1_y + max_mt_length_test * mt_dir_01_y;
% centro1_mt_ends_z   = centro1_z + max_mt_length_test * mt_dir_01_z;
%                                                   
% % MTs for the centrosome 2                                                   
% centro2_mt_ends_x   = centro2_x + max_mt_length_test * mt_dir_02_x;
% centro2_mt_ends_y   = centro2_y + max_mt_length_test * mt_dir_02_y;
% centro2_mt_ends_z   = centro2_z + max_mt_length_test * mt_dir_02_z;
% 

% % %  
% figure('position', [100   100   500   500]);
% hold on;
% axis equal;
% xlabel('x'); ylabel('y'); zlabel('z')
% view([-15 15])
% box on

% % shape
%  patch(isosurface(Shape_mat, 0.1), 'FaceColor', [0 1 1], 'FaceAlpha', 0.5, 'EdgeColor', 'none')
% 
% % nucleus center
% plot3(x, y, z, 'ko')
% 
% % spindle
% plot3([centro1_x, centro2_x], [centro1_y centro2_y], [centro1_z centro2_z], 'k-', 'linewidth', 3)
% plot3([centro1_x ], [centro1_y ], [centro1_z ], 'ro')
% plot3([centro2_x ], [centro2_y ], [centro2_z], 'bo')
% 
%  
% N = 20;
% for a = [1,N_alpha]%1 : N : N_alpha
%     if a==1
%         lw = 2;
%     else
%         lw = 1;
%     end
%    for p = 1 : N : N_psi
%        plot3([centro1_x, centro1_mt_ends_x(p, a)], [centro1_y, centro1_mt_ends_y(p, a)], [centro1_z, centro1_mt_ends_z(p, a)], '-', 'color', [1 0.8 0.8],'linewidth', lw)
%        plot3([centro2_x, centro2_mt_ends_x(p, a)], [centro2_y, centro2_mt_ends_y(p, a)], [centro2_z, centro2_mt_ends_z(p, a)], '-', 'color', [0.8 0.8 1],'linewidth', lw)
%    end
% end
%       
% 
% p = 1; a = 1; % spindle direction
% plot3([centro1_x, centro1_mt_ends_x(p, a)],[centro1_y, centro1_mt_ends_y(p, a)],[centro1_z, centro1_mt_ends_z(p, a)], 'r*-', 'linewidth',2)
% plot3([centro2_x, centro2_mt_ends_x(p, a)],[centro2_y, centro2_mt_ends_y(p, a)],[centro2_z, centro2_mt_ends_z(p, a)], 'b*-', 'linewidth',2)
% p = 1; a = 45; % psi 1
% plot3([centro1_x, centro1_mt_ends_x(p, a)],  [centro1_y, centro1_mt_ends_y(p, a)],[centro1_z, centro1_mt_ends_z(p, a)], 'ro-')
% plot3([centro2_x, centro2_mt_ends_x(p, a)], [centro2_y, centro2_mt_ends_y(p, a)], [centro2_z, centro2_mt_ends_z(p, a)], 'bo-')
% p = 45; a = 45; % psi 2
% plot3([centro1_x, centro1_mt_ends_x(p, a)],  [centro1_y, centro1_mt_ends_y(p, a)], [centro1_z, centro1_mt_ends_z(p, a)], 'rs-')
% plot3([centro2_x, centro2_mt_ends_x(p, a)], [centro2_y, centro2_mt_ends_y(p, a)], [centro2_z, centro2_mt_ends_z(p, a)], 'bs-')
% p = 90; a = 45; % psi 3
% plot3([centro1_x, centro1_mt_ends_x(p, a)], [centro1_y, centro1_mt_ends_y(p, a)], [centro1_z, centro1_mt_ends_z(p, a)], 'rd-')
% plot3([centro2_x, centro2_mt_ends_x(p, a)],  [centro2_y, centro2_mt_ends_y(p, a)], [centro2_z, centro2_mt_ends_z(p, a)], 'bd-')
% 
% len =  150;
% p = 1; a = 1; % spindle direction
% plot3(centro1_x + len * [0, mt_dir_01_x(p, a)], centro1_y + len * [0, mt_dir_01_y(p, a)], centro1_z + len * [0, mt_dir_01_z(p, a)], 'r*-', 'linewidth',2)
% plot3(centro2_x + len * [0, mt_dir_02_x(p, a)], centro2_y + len * [0, mt_dir_02_y(p, a)], centro2_z + len * [0, mt_dir_02_z(p, a)], 'b*-', 'linewidth',2)
% p = 1; a = 45; % psi 1
% plot3(centro1_x + len * [0, mt_dir_01_x(p, a)], centro1_y + len * [0, mt_dir_01_y(p, a)], centro1_z + len * [0, mt_dir_01_z(p, a)], 'ro-')
% plot3(centro2_x + len * [0, mt_dir_02_x(p, a)], centro2_y + len * [0, mt_dir_02_y(p, a)], centro2_z + len * [0, mt_dir_02_z(p, a)], 'bo-')
% p = 45; a = 45; % psi 2
% plot3(centro1_x + len * [0, mt_dir_01_x(p, a)], centro1_y + len * [0, mt_dir_01_y(p, a)], centro1_z + len * [0, mt_dir_01_z(p, a)], 'rs-')
% plot3(centro2_x + len * [0, mt_dir_02_x(p, a)], centro2_y + len * [0, mt_dir_02_y(p, a)], centro2_z + len * [0, mt_dir_02_z(p, a)], 'bs-')
% p = 90; a = 45; % psi 3
% plot3(centro1_x + len * [0, mt_dir_01_x(p, a)], centro1_y + len * [0, mt_dir_01_y(p, a)], centro1_z + len * [0, mt_dir_01_z(p, a)], 'rd-')
% plot3(centro2_x + len * [0, mt_dir_02_x(p, a)], centro2_y + len * [0, mt_dir_02_y(p, a)], centro2_z + len * [0, mt_dir_02_z(p, a)], 'bd-')



                                                  
                                                   
%% Correction for dependency of MT density on the polar angle Alpha.

% If it plot 180 MT  (1 mt per degree of psi) aroun the pole (alpha = 0),
% then the density of MT will be very high. However if you trace THE SAME number
% of MT (180) around equator (alpha = 90), the density is much lower.
% Thus, when you integrate the force aroun psi for a given Alpha,
% you get a bias of increasing the force with decrease in sin(alpha).

% Correction for this effect: basically, scale the force at each Alpha with
% Fcorrected = Fraw * sin(Alpha). However this will give you force = 0 at
% the alpha = 0 or 180.

% Why AP2016 had it like this:
% densMT                = (pi * 2 * (1 - cos(anglelimit_frac_of_180 * pi))); % Number of MTs per solid angle unit (1 rd2) 
% density_corr_ap2016   = densMT * pi/180*(abs(cos_alphamat-cos(alphamat_rad - pi / (180) ))); 

% the final result in general looks like sin(alpha)
% figure; imagesc(density_corr)
% figure; imagesc(sin_Alphamat)
% plot(sum(density_corr,1),'r-')
% But the scaling to 2*pi... is weird. Why?:


% density_corr_de2017  = sin_alphamat;


if 1
    % Another way, more "physically realistic"
    % lets just FOR REAL change the number of MT in psi direction, depending
    % on how high we ar ein Alpha. If alpha close to 0 or 180 - let's remove a
    % lot of MT, and if alpha -> 90, lets not remove any. The number of removed
    % MT will be proportional to Sin(Alpha), of course.

    % define the matrix, in which we will mark what MT to use.
    density_corr_de2017                = zeros(N_psi, N_alpha);

    % if I use sin(alpha) at 1 and last Column (alpha), then there will be NO mts.
    % instead, I want one MT at alpha = 0 and alpha = 180. % Let's force 1 MT
    % in these columns (column 1 and column end).
    density_corr_de2017(1, [1, end])   = 1;       

    % the rest columns we can calculate according to sin(alpha).
    for i = 2 : N_alpha - 1

        reduce_density_N_fold = 1 / sin(alpha_set_rad(i));
        take_each_Nth_element = round(1 : reduce_density_N_fold : N_psi);
        density_corr_de2017(take_each_Nth_element, i) = 1;
    end
end

% figure;
% hold on
% plot(sum(density_corr_de2017, 1), 'r-')
% plot(sum(sin_alphamat, 1), 'g-')





%% Collect MT length values, Stability factors and Polarity Domain values for each thrown MT

% preallocations; by default all contain only 0.
raw_length_per_mt_3D_01(N_psi, N_alpha, max_mt_length) = 0;%      = zeros(360, 180, max_mt_length);    % 3rd dimension: collecting MT length along MT
raw_length_per_mt_3D_02(N_psi, N_alpha, max_mt_length) = 0;%       = zeros(360, 180, max_mt_length);
stability_per_mt_3D_01(N_psi, N_alpha, max_mt_length) = 0;%        = zeros(360, 180, max_mt_length);    % 3rd dimension: collecting MT stability along MT
stability_per_mt_3D_02(N_psi, N_alpha, max_mt_length) = 0;%        = zeros(360, 180, max_mt_length);

polarity_per_mt_3D_01(N_psi, N_alpha, max_mt_length) = 0;%         = zeros(360, 180);                   % presense of polarity domain = 1; absense = 0.
polarity_per_mt_3D_02(N_psi, N_alpha, max_mt_length) = 0;%         = zeros(360, 180); 
%polarity_per_mt_2D_01(360, 180) = 0;%         = zeros(360, 180);                   % presense of polarity domain = 1; absense = 0.
%polarity_per_mt_2D_02(360, 180) = 0;%         = zeros(360, 180); 

% alpha - angle of a MT relative to the spindle orinetation.
% Exactly the same as Theta in spherical coordinates: 
% Theta = angle btw <z, a vector>,
% alpha = angle btw <spindle, a MT>



% [~, alpha_ind_max]       = min(abs(alpha_set_deg - anglelimit_frac_of_180 * 180));
% check it is close to the target angle:

for alpha_ind = 1 : N_alpha      % limit the alpha to the anglelimit. 
    
    for psi_ind = 1 : N_psi    
                
        % set of length elements; projecitons
        mt_elements_x       = linspace(centro1_x, centro1_mt_ends_x(psi_ind, alpha_ind), max_mt_length);
        mt_elements_y       = linspace(centro1_y, centro1_mt_ends_y(psi_ind, alpha_ind), max_mt_length);
        mt_elements_z       = linspace(centro1_z, centro1_mt_ends_z(psi_ind, alpha_ind), max_mt_length);
        
        % start checking with the first element (at the centrosome)
        el_N        = 1;
        el_x        = mt_elements_x(el_N);          % cartesian coords of the length element
        el_y        = mt_elements_y(el_N);
        el_z        = mt_elements_z(el_N);            
        el_x_ind    = floor(mt_elements_x(el_N));   % integer Cartesian coords of the length element. To be used in matrices.
        el_y_ind    = floor(mt_elements_y(el_N));
        el_z_ind    = floor(mt_elements_z(el_N));
        
        while Shape_mat(el_y_ind, el_x_ind, el_z_ind)   % if the MT element is inside the shape   
            
            stability_per_mt_3D_01(psi_ind, alpha_ind, el_N)  = Stab_mat(el_y_ind, el_x_ind, el_z_ind);

            %if calculating_polarity % faster without checking
                polarity_per_mt_3D_01(psi_ind, alpha_ind, el_N)   = Pol_mat(el_y_ind , el_x_ind , el_z_ind);
           % end
            
            raw_length_per_mt_3D_01(psi_ind, alpha_ind, el_N) = sqrt((el_x - centro1_x)^2 + ...
                                                                 (el_y - centro1_y)^2 + ...
                                                                 (el_z - centro1_z)^2 );
            % go to the next element
            el_N        = el_N + 1;            
            el_x        = mt_elements_x(el_N);          % cartesian coords of the length element
            el_y        = mt_elements_y(el_N);
            el_z        = mt_elements_z(el_N);            
            el_x_ind    = floor(mt_elements_x(el_N));   % integer Cartesian coords of the length element. To be used in matrices.
            el_y_ind    = floor(mt_elements_y(el_N));
            el_z_ind    = floor(mt_elements_z(el_N));
        end              

        % polarity_per_mt_3D_01(psi_ind, alpha_ind, :)
        
        % polarity:
%         if calculating_polarity
%             pol_factors         = zeros(el_N, 1);
%             for ll = 1 : el_N
%                  el_x_ind           = floor(mt_elements_x(ll));   % integer Cartesian coords of the length element. To be used in matrices.
%                  el_y_ind           = floor(mt_elements_y(ll));
%                  el_z_ind           = floor(mt_elements_z(ll));
%                  pol_factors(ll)    = Pol_mat(el_y_ind, el_x_ind, el_z_ind);        
%             end
%             polarity_per_mt_2D_01(psi, alpha) = max(pol_factors);
%         end

%         if 0
%             % show both MT ends:
%                % start end end of the MT
%             mt_end_1            = [centro1_x, centro1_y, centro1_z];
%             mt_end_2            = [centro1_mt_ends_x(psi, alpha), centro1_mt_ends_y(psi, alpha), centro1_mt_ends_z(psi, alpha)];               
%         
%             plot3([mt_end_1(1), mt_end_2(1)], ...
%                   [mt_end_1(2), mt_end_2(2)], ...
%                   [mt_end_1(3), mt_end_2(3)], 'ks--', 'markersize',15)
%              
%             plot3(mt_elements_x, ...
%                   mt_elements_y, ...
%                   mt_elements_z, 'ks', 'markersize',4)
%            
%               
%             % Stability along the current MT
%             figure; 
%             hold on;
%             xlabel('mt length (pix)')
%            
%             % Stability
%             plot(mt_in_len_stab(mt_in_len_stab(:, 1) == 1, 2), ...
%                  mt_in_len_stab(mt_in_len_stab(:, 1) == 1, 3), 'go')           
%             legend('stability')              
%         end
        

        % ===========================================    for 2nd aster  ===  
        % set of length elements; projecitons
        mt_elements_x       = linspace(centro2_x, centro2_mt_ends_x(psi_ind, alpha_ind), max_mt_length);
        mt_elements_y       = linspace(centro2_y, centro2_mt_ends_y(psi_ind, alpha_ind), max_mt_length);
        mt_elements_z       = linspace(centro2_z, centro2_mt_ends_z(psi_ind, alpha_ind), max_mt_length);
        
        % start checking with the first element (at the centrosome)
        el_N        = 1;
        el_x        = mt_elements_x(el_N);          % cartesian coords of the length element
        el_y        = mt_elements_y(el_N);
        el_z        = mt_elements_z(el_N);            
        el_x_ind    = floor(mt_elements_x(el_N));   % integer Cartesian coords of the length element. To be used in matrices.
        el_y_ind    = floor(mt_elements_y(el_N));
        el_z_ind    = floor(mt_elements_z(el_N));
        
        while Shape_mat(el_y_ind, el_x_ind, el_z_ind)   % if the MT element is inside the shape   
            
            stability_per_mt_3D_02(psi_ind, alpha_ind, el_N)  = Stab_mat(el_y_ind, el_x_ind, el_z_ind);
            
            %if calculating_polarity % faster without checking
                polarity_per_mt_3D_02(psi_ind, alpha_ind, el_N)   = Pol_mat(el_y_ind , el_x_ind , el_z_ind);
            %end
            
            raw_length_per_mt_3D_02(psi_ind, alpha_ind, el_N) = sqrt((el_x - centro2_x)^2 + ...
                                                                     (el_y - centro2_y)^2 + ...
                                                                     (el_z - centro2_z)^2 );

            % go to the next element
            el_N        = el_N + 1;            
            el_x        = mt_elements_x(el_N);          % cartesian coords of the length element
            el_y        = mt_elements_y(el_N);
            el_z        = mt_elements_z(el_N);            
            el_x_ind    = floor(mt_elements_x(el_N));   % integer Cartesian coords of the length element. To be used in matrices.
            el_y_ind    = floor(mt_elements_y(el_N));
            el_z_ind    = floor(mt_elements_z(el_N));
        end  
    end
end





%% Finalalze matrices for Length 
% Used the same idea as in "trapz(X,Y)" function, but in 3rd dimension:

%  -----------------------------    Lengths of MT from centrosome 1 -------
xx       = diff(raw_length_per_mt_3D_01, 1, 3);                                    % each X element is MT length element size.
yy       = stability_per_mt_3D_01(:,:, 1:end-1) + diff(stability_per_mt_3D_01, 1, 3)/2;      % each Y element is [y(i) + y(i+1)]/2

yy(xx < 0)  = 0;    % remove junk (negative lengths due to Diff function)
xx(xx < 0)  = 0;    % remove junk (wrong stabs due to Diff function)

length_stab_corr_1  = sum(xx .* yy, 3);                           % stability corrected length  = sum (Xbin * Y) over all elements X Exactly as trapz(x,y).

% -----------------------------    Lengths of MT from centrosome 2 -------
xx       = diff(raw_length_per_mt_3D_02, 1, 3);                                    % each X element is MT length element size.
yy       = stability_per_mt_3D_02(:,:, 1:end-1) + diff(stability_per_mt_3D_02, 1, 3)/2;      % each Y element is [y(i) + y(i+1)]/2

yy(xx < 0)  = 0;    % remove junk (negative lengths due to Diff function)
xx(xx < 0)  = 0;    % remove junk (wrong stabs due to Diff function)

length_stab_corr_2  = sum(xx .* yy, 3);                           % stability corrected length  = sum (Xbin * Y) over all elements X Exactly as trapz(x,y).



% -----------------------------------       scale to [0-1]   ----------
length_final_01    = length_stab_corr_1 / shape_span_avg;      % stab. corr and rescaled to [0, 1] (almost)
length_final_02    = length_stab_corr_2 / shape_span_avg;      % stab. corr and rescaled to [0, 1] (almost)



% 
% %DE DEBUG
% length_raw_1
% length_stab_corr_1
% length_final_01
% 
% length_raw_1       = max(raw_length_per_mt_3D_01, [], 3);         % real, uncorrected lengths; just for checking.
% length_raw_2       = max(raw_length_per_mt_3D_02, [], 3);         % real, uncorrected lengths; just for checking.
% 
% % % MTs for the centrosome 1  
% real_mt_ends_1_x   = centro1_x + length_raw_1 .* mt_dir_01_x;
% real_mt_ends_1_y   = centro1_y + length_raw_1 .* mt_dir_01_y;
% real_mt_ends_1_z   = centro1_z + length_raw_1 .* mt_dir_01_z;
%                                                   
% % MTs for the centrosome 2                                                   
% real_mt_ends_2_x   = centro2_x + length_raw_2 .* mt_dir_02_x;
% real_mt_ends_2_y   = centro2_y + length_raw_2 .* mt_dir_02_y;
% real_mt_ends_2_z   = centro2_z + length_raw_2 .* mt_dir_02_z;

%                         
%            
% figure('position', [100   100   600   600]);
% hold on;
% axis equal;
% axis tight
% xlabel('x'); ylabel('y'); zlabel('z')
% view([-15 15])
% box on

% % shape
%  patch(isosurface(Shape_mat,0.5), 'FaceColor', [0 1 1], 'FaceAlpha', 0.1, 'EdgeColor', 'none')
% 
% % nucle center
% plot3(x, y, z, 'ko')
% 
% % spindle
% plot3([centro1_x, centro2_x], [centro1_y centro2_y], [centro1_z centro2_z], 'k-', 'linewidth', 3)
% plot3([centro1_x ], [centro1_y ], [centro1_z ], 'ro')
% plot3([centro2_x ], [centro2_y ], [centro2_z], 'bo')
% 
%  
% N = 20;
% for a = 45%length(alpha_set_deg)
%    for p = 1:length(psi_set_deg)
%        plot3([centro1_x, real_mt_ends_1_x(p, a)], [centro1_y, real_mt_ends_1_y(p, a)], [centro1_z, real_mt_ends_1_z(p, a)], '-', 'color', [1 0.8 0.8])
%        plot3([centro2_x, real_mt_ends_2_x(p, a)], [centro2_y, real_mt_ends_2_y(p, a)], [centro2_z, real_mt_ends_2_z(p, a)], '-', 'color', [0.8 0.8 1])
%    end
% end
% 
% p = 1; a = 1; % spindle direction
% plot3([centro1_x, real_mt_ends_1_x(p, a)], [centro1_y, real_mt_ends_1_y(p, a)], [centro1_z, real_mt_ends_1_z(p, a)], 'r*-', 'linewidth',2)
% plot3([centro2_x, real_mt_ends_2_x(p, a)], [centro2_y, real_mt_ends_2_y(p, a)], [centro2_z, real_mt_ends_2_z(p, a)], 'b*-', 'linewidth',2)
% p = 1; a = 45; % psi 1
% plot3([centro1_x, real_mt_ends_1_x(p, a)], [centro1_y, real_mt_ends_1_y(p, a)], [centro1_z, real_mt_ends_1_z(p, a)], 'ro-')
% plot3([centro2_x, real_mt_ends_2_x(p, a)], [centro2_y, real_mt_ends_2_y(p, a)], [centro2_z, real_mt_ends_2_z(p, a)], 'bo-')
% p = 46; a = 45; % psi 2
% plot3([centro1_x, real_mt_ends_1_x(p, a)], [centro1_y, real_mt_ends_1_y(p, a)], [centro1_z, real_mt_ends_1_z(p, a)], 'rs-')
% plot3([centro2_x, real_mt_ends_2_x(p, a)], [centro2_y, real_mt_ends_2_y(p, a)], [centro2_z, real_mt_ends_2_z(p, a)], 'bs-')
% p = 90; a = 45; % psi 3
% plot3([centro1_x, real_mt_ends_1_x(p, a)], [centro1_y, real_mt_ends_1_y(p, a)], [centro1_z, real_mt_ends_1_z(p, a)], 'rd-')
% plot3([centro2_x, real_mt_ends_2_x(p, a)], [centro2_y, real_mt_ends_2_y(p, a)], [centro2_z, real_mt_ends_2_z(p, a)], 'bd-')
% 
% 
% % DIRECTIONAL VECTORS
% 
% len =  50;
% lw = 2;
% p = 1; a = 1; % spindle direction
% plot3(centro1_x + len * [0, mt_dir_01_x(p, a)], centro1_y + len * [0, mt_dir_01_y(p, a)], centro1_z + len * [0, mt_dir_01_z(p, a)], 'r*-', 'linewidth', lw)
% plot3(centro2_x + len * [0, mt_dir_02_x(p, a)], centro2_y + len * [0, mt_dir_02_y(p, a)], centro2_z + len * [0, mt_dir_02_z(p, a)], 'b*-', 'linewidth', lw)
% p = 1; a = 45; % psi 1
% plot3(centro1_x + len * [0, mt_dir_01_x(p, a)], centro1_y + len * [0, mt_dir_01_y(p, a)], centro1_z + len * [0, mt_dir_01_z(p, a)], 'ro-', 'linewidth', lw)
% plot3(centro2_x + len * [0, mt_dir_02_x(p, a)], centro2_y + len * [0, mt_dir_02_y(p, a)], centro2_z + len * [0, mt_dir_02_z(p, a)], 'bo-', 'linewidth', lw)
% p = 45; a = 45; % psi 2
% plot3(centro1_x + len * [0, mt_dir_01_x(p, a)], centro1_y + len * [0, mt_dir_01_y(p, a)], centro1_z + len * [0, mt_dir_01_z(p, a)], 'rs-', 'linewidth', lw)
% plot3(centro2_x + len * [0, mt_dir_02_x(p, a)], centro2_y + len * [0, mt_dir_02_y(p, a)], centro2_z + len * [0, mt_dir_02_z(p, a)], 'bs-', 'linewidth', lw)
% p = 90; a = 45; % psi 3
% plot3(centro1_x + len * [0, mt_dir_01_x(p, a)], centro1_y + len * [0, mt_dir_01_y(p, a)], centro1_z + len * [0, mt_dir_01_z(p, a)], 'rd-', 'linewidth', lw)
% plot3(centro2_x + len * [0, mt_dir_02_x(p, a)], centro2_y + len * [0, mt_dir_02_y(p, a)], centro2_z + len * [0, mt_dir_02_z(p, a)], 'bd-', 'linewidth', lw)
% 



%% Finalaze matrices for Polarity

% AP2016 also thresholds the polarity matrices, like that:
% p02_smooth = ceil((min(p02_smooth,1))-seuil_pol)
% she also scales them to [0,1] in  a weird way.

% One can bring back the rescaling to [0,1],
% p01_smooth = mat2gray(p01_smooth);
% p02_smooth = mat2gray(p02_smooth);
% but it is not needed actually; the matrices are [0,1] already...

% Fix the polarity matrices: 
%   - fill the holes
if calculating_polarity
        
    % max projection of polarity 3D should contain 1, if a MT has hit the
    % polarity domain.
    p01_final = max(polarity_per_mt_3D_01, [], 3);
    p02_final = max(polarity_per_mt_3D_02, [], 3);

    % one should close the wholes here.
    % But i had a very thick polarity domain, 3 pixels inwards.
    % this makes it impossible for any MT to escape between the pixels.
    % the polarity cortex thickness it set in the function
    % "set_polarity_domain_interactively".
    
    % close tiny holes - bad way, gives wrong results for large, concave
    % polarity domain (fill holes - will make everything a polar domain)
    % p01_final = imfill(polarity_per_mt_2D_01, 'holes');
    % p02_final = imfill(polarity_per_mt_2D_02, 'holes');

    % another way, a bit slower though:
    % p01_final = double(imgaussfilt(polarity_per_mt_2D_01, 1) > 0);
    % p02_final = double(imgaussfilt(polarity_per_mt_2D_02, 1) > 0);  
end   
    

% DEDEBUG: check
% length_stab_corr_1
% length_stab_corr_2
% p01_final
% p02_final




%% Calculation of force per each MT
% Forces are calculated according to this template:
% F = L^Beta1 + pullfactor*L^Beta2 + pullfactor_exp*2^(L/Unit_branch):

if calculating_polarity
    abs_forces_aster_1  =  length_final_01 .^ Beta1 + ...
                           pullfactor * p01_final .* length_final_01 .^ Beta2 + ...
                           pullfactor_exp * p01_final .* (2 .^ (length_final_01 / Unit_branch_2));
               
    abs_forces_aster_2  =  length_final_02 .^ Beta1 + ...
                           pullfactor * p02_final .* length_final_02 .^ Beta2 + ...
                           pullfactor_exp * p02_final .* (2 .^ (length_final_02 / Unit_branch_2));  
else
    abs_forces_aster_1  =  length_final_01 .^ Beta1;
    abs_forces_aster_2  =  length_final_02 .^ Beta1;    
end

% Coorect for angular density inconsistency
% AP2016 style.
% IF YOU WANT TO USE AP2016 DENSITY CORRECTION, YOU NEED TO UNCOMMENT
% THE LINES ABOVE THAT CONTAIN CALULATION OF density_corr_ap2016
% abs_forces_aster_1     = density_corr_ap2016 .* abs_forces_aster_1;         
% abs_forces_aster_2     = density_corr_ap2016 .* abs_forces_aster_2;

% DE2017 way:
% kick off some MT to correct for the density effect of Alpha by using mask
% density_corr_de2017
%warning('no density correction')
abs_forces_aster_1      = density_corr_de2017 .* abs_forces_aster_1;
abs_forces_aster_2      = density_corr_de2017 .* abs_forces_aster_2;




%% abs_forces_aster_1; abs_forces_aster_2

% Total force on the centrosome 1
f1_per_mt_x             = abs_forces_aster_1 .* mt_dir_01_x;
f1_per_mt_y             = abs_forces_aster_1 .* mt_dir_01_y;
f1_per_mt_z             = abs_forces_aster_1 .* mt_dir_01_z;
f_centro1_x             = sum(f1_per_mt_x(:));
f_centro1_y             = sum(f1_per_mt_y(:));
f_centro1_z             = sum(f1_per_mt_z(:));


% Total force on the centrosome 1
f2_per_mt_x             = abs_forces_aster_2 .* mt_dir_02_x;
f2_per_mt_y             = abs_forces_aster_2 .* mt_dir_02_y;
f2_per_mt_z             = abs_forces_aster_2 .* mt_dir_02_z;
f_centro2_x             = sum(f2_per_mt_x(:));
f_centro2_y             = sum(f2_per_mt_y(:));
f_centro2_z             = sum(f2_per_mt_z(:));

% force on the center of the spindle
f_nucl_x                = f_centro1_x + f_centro2_x;
f_nucl_y                = f_centro1_y + f_centro2_y;
f_nucl_z                = f_centro1_z + f_centro2_z;


% Projections of MT forces on two local (spindle) axes:
% Theta and Phi.
% these are absolute values of forces that will create torque along
% corresponding axes:
f1_per_mt_proj_theta     = abs_forces_aster_1 .* sin_alphamat .* cos_psimat;
f1_per_mt_proj_phi       = abs_forces_aster_1 .* sin_alphamat .* sin_psimat;

f2_per_mt_proj_theta     = abs_forces_aster_2 .* sin_alphamat .* cos_psimat;
f2_per_mt_proj_phi       = abs_forces_aster_2 .* sin_alphamat .* sin_psimat;

% instead of using cross (R, F) we can do away with scalars, however
% we should be very careful about - and +; thus:
total_torque_theta       = sum(f1_per_mt_proj_theta(:)) - sum(f2_per_mt_proj_theta(:)); % MINUS!!!
total_torque_phi         = sum(f1_per_mt_proj_phi(:))   + sum(f2_per_mt_proj_phi(:));
% IMPORTANT NOTE ON - and +:
% + or - between torques depends on HOW WE CHOOSE ORIENTATION OF THE
% TWO HALVES OF THE SPINLDE. WE CHOSE THEM ABOVE:
% THETA1 = theta, PHI1 = phi
% THETA2 = pi - THETA, PHI2 = phi + pi.
% In such configuration:
% positive f1_per_mt_proj_theta rotates to +theta
% positive f2_per_mt_proj_theta rotates to -theta. THUS "-" !!!
% positive f1_per_mt_proj_phi rotates to +phi
% positive f2_per_mt_proj_phi rotates to +phi. THUS "+" !!!
% 
torques_forces      = zeros(1,5);
torques_forces(1)   = total_torque_theta;
torques_forces(2)   = total_torque_phi;
torques_forces(3)   = f_nucl_x;
torques_forces(4)   = f_nucl_y;
torques_forces(5)   = f_nucl_z;



% figure('position', [100   100   600   600]);
% hold on;
% axis equal;
% axis tight
% xlabel('x'); ylabel('y'); zlabel('z')
% view([-15 15])
% box on


% shape
% patch(isosurface(Shape_mat,0.5), 'FaceColor', [0 1 1], 'FaceAlpha', 0.5, 'EdgeColor', 'none')
% 
% % nucleus center
% plot3(x, y, z, 'ko')
% 
% % spindle
% plot3([centro1_x, centro2_x], [centro1_y centro2_y], [centro1_z centro2_z], 'k-', 'linewidth', 3)
% plot3([centro1_x ], [centro1_y ], [centro1_z ], 'ro')
% plot3([centro2_x ], [centro2_y ], [centro2_z], 'bo')

% some MT
% N = 1;
% for a = 45%length(alpha_set_deg)
%    for p = 1 : N : length(psi_set_deg)
%        plot3([centro1_x, real_mt_ends_1_x(p, a)], [centro1_y, real_mt_ends_1_y(p, a)], [centro1_z, real_mt_ends_1_z(p, a)], '-', 'color', [1 0.8 0.8])
%        plot3([centro2_x, real_mt_ends_2_x(p, a)], [centro2_y, real_mt_ends_2_y(p, a)], [centro2_z, real_mt_ends_2_z(p, a)], '-', 'color', [0.8 0.8 1])
%    end
% end
% 
% % some FORCES:
% lw = 2;
% sc1 = 1000;
% sc2 = 200;
% N = 1;
% for a = length(alpha_set_deg)-1
%    for p = 1 : N : length(psi_set_deg)
%        plot3(centro1_x + sc1*[0, f1_per_mt_x(p, a)], centro1_y + sc1*[0, f1_per_mt_y(p, a)], centro1_z + sc1*[0, f1_per_mt_z(p, a)], '-', 'color', [1 0.8 0.8], 'linewidth', lw)
%        plot3(centro2_x + sc2*[0, f2_per_mt_x(p, a)], centro2_y + sc2*[0, f2_per_mt_y(p, a)], centro2_z + sc2*[0, f2_per_mt_z(p, a)], '-', 'color', [0.8 0.8 1], 'linewidth', lw)
%    end
% end

% TOTAL FORCES:
% lw = 2;
% sc1 = 0.1;
% sc2 = 0.1;
% sc3 = 0.1;
% plot3(centro1_x + sc1*[0, f_centro1_x], centro1_y + sc1*[0, f_centro1_y], centro1_z + sc1*[0, f_centro1_z], '-', 'color', [1 0 0], 'linewidth', lw)
% plot3(centro2_x + sc2*[0, f_centro2_x], centro2_y + sc2*[0, f_centro2_y], centro2_z + sc2*[0, f_centro2_z], '-', 'color', [0. 0 1], 'linewidth', lw)
% plot3(x + sc3*[0, f_nucl_x], y + sc3*[0, f_nucl_y], z + sc3*[0, f_nucl_z], '-', 'color', 0* [ 1 1 1], 'linewidth', lw)


% SOME DIRECTIONAL VECTORS
% len =  250;
% lw = 1;
% p = 1; a = 1; % spindle direction
% plot3(centro1_x + len * [0, mt_dir_01_x(p, a)], centro1_y + len * [0, mt_dir_01_y(p, a)], centro1_z + len * [0, mt_dir_01_z(p, a)], 'r*-', 'linewidth', lw)
% plot3(centro2_x + len * [0, mt_dir_02_x(p, a)], centro2_y + len * [0, mt_dir_02_y(p, a)], centro2_z + len * [0, mt_dir_02_z(p, a)], 'b*-', 'linewidth', lw)
% p = 1; a = 45; % psi 1
% plot3(centro1_x + len * [0, mt_dir_01_x(p, a)], centro1_y + len * [0, mt_dir_01_y(p, a)], centro1_z + len * [0, mt_dir_01_z(p, a)], 'ro-', 'linewidth', lw)
% plot3(centro2_x + len * [0, mt_dir_02_x(p, a)], centro2_y + len * [0, mt_dir_02_y(p, a)], centro2_z + len * [0, mt_dir_02_z(p, a)], 'bo-', 'linewidth', lw)
% p = 45; a = 45; % psi 2
% plot3(centro1_x + len * [0, mt_dir_01_x(p, a)], centro1_y + len * [0, mt_dir_01_y(p, a)], centro1_z + len * [0, mt_dir_01_z(p, a)], 'rs-', 'linewidth', lw)
% plot3(centro2_x + len * [0, mt_dir_02_x(p, a)], centro2_y + len * [0, mt_dir_02_y(p, a)], centro2_z + len * [0, mt_dir_02_z(p, a)], 'bs-', 'linewidth', lw)
% p = 90; a = 45; % psi 3
% plot3(centro1_x + len * [0, mt_dir_01_x(p, a)], centro1_y + len * [0, mt_dir_01_y(p, a)], centro1_z + len * [0, mt_dir_01_z(p, a)], 'rd-', 'linewidth', lw)
% plot3(centro2_x + len * [0, mt_dir_02_x(p, a)], centro2_y + len * [0, mt_dir_02_y(p, a)], centro2_z + len * [0, mt_dir_02_z(p, a)], 'bd-', 'linewidth', lw)
% 







% -----------------------------   Torques    ------------------------------
% (along theta and phi) for all MT, for asters 1 (MT1) and 2 (MT2)
% torqueperMT1(:, :, 1)   =  abs_forces_aster_1 .* sin_alphamat .* sin_psimat;  % Torque along theta, aster 1
% torqueperMT1(:, :, 2)   = -abs_forces_aster_1 .* sin_alphamat .* cos_psimat;  % Torque along phi, aster 1                     
% torqueperMT2(:, :, 1)   =  abs_forces_aster_2 .* sin_alphamat .* sin_psimat;  % Torque along theta, aster 2
% torqueperMT2(:, :, 2)   = -abs_forces_aster_2 .* sin_alphamat .* cos_psimat;  % Torque along phi, aster 2
% 
% torqueperMT1(:, :, 1)   =  abs_forces_aster_1 .* sin_alphamat .* cos_psimat;  % Torque along theta, aster 1
% torqueperMT1(:, :, 2)   =  abs_forces_aster_1 .* sin_alphamat .* sin_psimat;  % Torque along phi, aster 1      
% 
% torqueperMT2(:, :, 1)   =  abs_forces_aster_2 .* sin_alphamat .* cos_psimat;  % Torque along theta, aster 2
% torqueperMT2(:, :, 2)   =  abs_forces_aster_2 .* sin_alphamat .* sin_psimat;  % Torque along phi, aster 2



% -----------------------------    Forces    ------------------------------
% (directions x, y, z are RELATIVE TO THE SPINDLE ORIENTATION) for all MT, for asters 1 and 2
% aster 1
% forceperMT1z(:, :)      =   abs_forces_aster_1 .* cos_alphamat;                % z: along the spindle                       
% forceperMT1x(:, :)      =   abs_forces_aster_1 .* sin_alphamat .* cos_psimat;                       
% forceperMT1y(:, :)      =   abs_forces_aster_1 .* sin_alphamat .* sin_psimat; 
%                          
% % aster 2     
% forceperMT2z(:, :)      = - abs_forces_aster_2 .* cos_alphamat;                % z: along the spindle 
% forceperMT2x(:, :)      = - abs_forces_aster_2 .* sin_alphamat.*cos_psimat;                        
% forceperMT2y(:, :)      = - abs_forces_aster_2 .* sin_alphamat.*sin_psimat;

% DE2017DEBUG: AP2016 had it exactly like this:
% Net torque (along theta and phi) and force (along x, y, z)
% Potentiel(it,1)=(sum(sum(torqueperMT1(:,1:floor(anglelimit*180),1)))+sum(sum(torqueperMT2(:,1:floor(anglelimit*180),1))));
% Potentiel(it,2)=(sum(sum(torqueperMT1(:,1:floor(anglelimit*180),2)))+sum(sum(torqueperMT2(:,1:floor(anglelimit*180),2))));
% pot3=sum(sum(forceperMT1x(:,1:floor(anglelimit*180))))+sum(sum(forceperMT2x(:,1:floor(anglelimit*180))));
% pot4=sum(sum(forceperMT1y(:,1:floor(anglelimit*180))))+sum(sum(forceperMT2y(:,1:floor(anglelimit*180))));
% pot5=sum(sum(forceperMT1z(:,1:floor(anglelimit*180))))+sum(sum(forceperMT2z(:,1:floor(anglelimit*180))));
% Potentiel(it,3)=(pot5*sin(thetarad)*cos(phirad)-pot4*sin(phirad)+pot3*cos(thetarad)*cos(phirad));
% Potentiel(it,4)=(pot5*sin(thetarad)*sin(phirad)+pot4*cos(phirad)+pot3*cos(thetarad)*sin(phirad));
% Potentiel(it,5)=pot5*cos(thetarad)-pot3*sin(thetarad);
% pottempa=Potentiel(it,:);       % net torque and force matrix


% =========================================  output adaptor starts ========
% Let's reshape the paragraph above just a bit to adapt the output to the "sim_program".

% Net torque (nt) along theta and phi; same as in the paragraph above:
% Potentiel(it,1)=(sum(sum(torqueperMT1(:,1:floor(anglelimit*180),1)))+sum(sum(torqueperMT2(:,1:floor(anglelimit*180),1))));
% Potentiel(it,2)=(sum(sum(torqueperMT1(:,1:floor(anglelimit*180),2)))+sum(sum(torqueperMT2(:,1:floor(anglelimit*180),2))));
% pot1        = (sum(sum(torqueperMT1(:,1:alpha_ind_max,1))) + sum(sum(torqueperMT2(:,1:alpha_ind_max,1))));
% pot2        = (sum(sum(torqueperMT1(:,1:alpha_ind_max,2))) + sum(sum(torqueperMT2(:,1:alpha_ind_max,2))));    
% 
% % Net force along x, y, z; same as in the paragraph above:
% pot3        = sum(sum(forceperMT1x(:, 1:alpha_ind_max))) + sum(sum(forceperMT2x(:, 1:alpha_ind_max)));
% pot4        = sum(sum(forceperMT1y(:, 1:alpha_ind_max))) + sum(sum(forceperMT2y(:, 1:alpha_ind_max)));
% pot5        = sum(sum(forceperMT1z(:, 1:alpha_ind_max))) + sum(sum(forceperMT2z(:, 1:alpha_ind_max)));    
% 
% 
% 
% % Final output:
% % torque and force matrix
% torques_forces    = nan(1,5);
% torques_forces(1) = pot1;
% torques_forces(2) = pot2;
% % translate from spindle coordinates to Cartesian coordinates; same as in the paragraph above:
% % Potentiel(it,3)=(pot5*sin(thetarad)*cos(phirad)-pot4*sin(phirad)+pot3*cos(thetarad)*cos(phirad));
% % Potentiel(it,4)=(pot5*sin(thetarad)*sin(phirad)+pot4*cos(phirad)+pot3*cos(thetarad)*sin(phirad));
% % Potentiel(it,5)=pot5*cos(thetarad)-pot3*sin(thetarad);
% torques_forces(3) = (pot5*sin(theta_rad)*cos(phi_rad) - pot4*sin(phi_rad) + pot3*cos(theta_rad)*cos(phi_rad));
% torques_forces(4) = (pot5*sin(theta_rad)*sin(phi_rad) + pot4*cos(phi_rad) + pot3*cos(theta_rad)*sin(phi_rad));
% torques_forces(5) =  pot5*cos(theta_rad) - pot3*sin(theta_rad);
% =========================================  output adaptor finishes ======

%toc

end


